/*! firebase-admin v8.7.0 */
"use strict";
/*!
 * Copyright 2017 Google Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
var credential_1 = require("./auth/credential");
var validator = require("./utils/validator");
var deep_copy_1 = require("./utils/deep-copy");
var error_1 = require("./utils/error");
/**
 * Internals of a FirebaseApp instance.
 */
var FirebaseAppInternals = /** @class */ (function () {
    function FirebaseAppInternals(credential_) {
        this.credential_ = credential_;
        this.isDeleted_ = false;
        this.tokenListeners_ = [];
    }
    /**
     * Gets an auth token for the associated app.
     *
     * @param {boolean} forceRefresh Whether or not to force a token refresh.
     * @return {Promise<FirebaseAccessToken>} A Promise that will be fulfilled with the current or
     *   new token.
     */
    FirebaseAppInternals.prototype.getToken = function (forceRefresh) {
        var _this = this;
        var expired = this.cachedToken_ && this.cachedToken_.expirationTime < Date.now();
        if (this.cachedTokenPromise_ && !forceRefresh && !expired) {
            return this.cachedTokenPromise_
                .catch(function (error) {
                // Update the cached token promise to avoid caching errors. Set it to resolve with the
                // cached token if we have one (and return that promise since the token has still not
                // expired).
                if (_this.cachedToken_) {
                    _this.cachedTokenPromise_ = Promise.resolve(_this.cachedToken_);
                    return _this.cachedTokenPromise_;
                }
                // Otherwise, set the cached token promise to null so that it will force a refresh next
                // time getToken() is called.
                _this.cachedTokenPromise_ = null;
                // And re-throw the caught error.
                throw error;
            });
        }
        else {
            // Clear the outstanding token refresh timeout. This is a noop if the timeout is undefined.
            clearTimeout(this.tokenRefreshTimeout_);
            // this.credential_ may be an external class; resolving it in a promise helps us
            // protect against exceptions and upgrades the result to a promise in all cases.
            this.cachedTokenPromise_ = Promise.resolve(this.credential_.getAccessToken())
                .then(function (result) {
                // Since the developer can provide the credential implementation, we want to weakly verify
                // the return type until the type is properly exported.
                if (!validator.isNonNullObject(result) ||
                    typeof result.expires_in !== 'number' ||
                    typeof result.access_token !== 'string') {
                    throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_CREDENTIAL, "Invalid access token generated: \"" + JSON.stringify(result) + "\". Valid access " +
                        'tokens must be an object with the "expires_in" (number) and "access_token" ' +
                        '(string) properties.');
                }
                var token = {
                    accessToken: result.access_token,
                    expirationTime: Date.now() + (result.expires_in * 1000),
                };
                var hasAccessTokenChanged = (_this.cachedToken_ && _this.cachedToken_.accessToken !== token.accessToken);
                var hasExpirationChanged = (_this.cachedToken_ && _this.cachedToken_.expirationTime !== token.expirationTime);
                if (!_this.cachedToken_ || hasAccessTokenChanged || hasExpirationChanged) {
                    _this.cachedToken_ = token;
                    _this.tokenListeners_.forEach(function (listener) {
                        listener(token.accessToken);
                    });
                }
                // Establish a timeout to proactively refresh the token every minute starting at five
                // minutes before it expires. Once a token refresh succeeds, no further retries are
                // needed; if it fails, retry every minute until the token expires (resulting in a total
                // of four retries: at 4, 3, 2, and 1 minutes).
                var refreshTimeInSeconds = (result.expires_in - (5 * 60));
                var numRetries = 4;
                // In the rare cases the token is short-lived (that is, it expires in less than five
                // minutes from when it was fetched), establish the timeout to refresh it after the
                // current minute ends and update the number of retries that should be attempted before
                // the token expires.
                if (refreshTimeInSeconds <= 0) {
                    refreshTimeInSeconds = result.expires_in % 60;
                    numRetries = Math.floor(result.expires_in / 60) - 1;
                }
                // The token refresh timeout keeps the Node.js process alive, so only create it if this
                // instance has not already been deleted.
                if (numRetries && !_this.isDeleted_) {
                    _this.setTokenRefreshTimeout(refreshTimeInSeconds * 1000, numRetries);
                }
                return token;
            })
                .catch(function (error) {
                var errorMessage = (typeof error === 'string') ? error : error.message;
                errorMessage = 'Credential implementation provided to initializeApp() via the ' +
                    '"credential" property failed to fetch a valid Google OAuth2 access token with the ' +
                    ("following error: \"" + errorMessage + "\".");
                if (errorMessage.indexOf('invalid_grant') !== -1) {
                    errorMessage += ' There are two likely causes: (1) your server time is not properly ' +
                        'synced or (2) your certificate key file has been revoked. To solve (1), re-sync the ' +
                        'time on your server. To solve (2), make sure the key ID for your key file is still ' +
                        'present at https://console.firebase.google.com/iam-admin/serviceaccounts/project. If ' +
                        'not, generate a new key file at ' +
                        'https://console.firebase.google.com/project/_/settings/serviceaccounts/adminsdk.';
                }
                throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_CREDENTIAL, errorMessage);
            });
            return this.cachedTokenPromise_;
        }
    };
    /**
     * Adds a listener that is called each time a token changes.
     *
     * @param {function(string)} listener The listener that will be called with each new token.
     */
    FirebaseAppInternals.prototype.addAuthTokenListener = function (listener) {
        this.tokenListeners_.push(listener);
        if (this.cachedToken_) {
            listener(this.cachedToken_.accessToken);
        }
    };
    /**
     * Removes a token listener.
     *
     * @param {function(string)} listener The listener to remove.
     */
    FirebaseAppInternals.prototype.removeAuthTokenListener = function (listener) {
        this.tokenListeners_ = this.tokenListeners_.filter(function (other) { return other !== listener; });
    };
    /**
     * Deletes the FirebaseAppInternals instance.
     */
    FirebaseAppInternals.prototype.delete = function () {
        this.isDeleted_ = true;
        // Clear the token refresh timeout so it doesn't keep the Node.js process alive.
        clearTimeout(this.tokenRefreshTimeout_);
    };
    /**
     * Establishes timeout to refresh the Google OAuth2 access token used by the SDK.
     *
     * @param {number} delayInMilliseconds The delay to use for the timeout.
     * @param {number} numRetries The number of times to retry fetching a new token if the prior fetch
     *   failed.
     */
    FirebaseAppInternals.prototype.setTokenRefreshTimeout = function (delayInMilliseconds, numRetries) {
        var _this = this;
        this.tokenRefreshTimeout_ = setTimeout(function () {
            _this.getToken(/* forceRefresh */ true)
                .catch(function (error) {
                // Ignore the error since this might just be an intermittent failure. If we really cannot
                // refresh the token, an error will be logged once the existing token expires and we try
                // to fetch a fresh one.
                if (numRetries > 0) {
                    _this.setTokenRefreshTimeout(60 * 1000, numRetries - 1);
                }
            });
        }, delayInMilliseconds);
    };
    return FirebaseAppInternals;
}());
exports.FirebaseAppInternals = FirebaseAppInternals;
/**
 * Global context object for a collection of services using a shared authentication state.
 */
var FirebaseApp = /** @class */ (function () {
    function FirebaseApp(options, name, firebaseInternals_) {
        var _this = this;
        this.firebaseInternals_ = firebaseInternals_;
        this.services_ = {};
        this.isDeleted_ = false;
        this.name_ = name;
        this.options_ = deep_copy_1.deepCopy(options);
        if (!validator.isNonNullObject(this.options_)) {
            throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_APP_OPTIONS, "Invalid Firebase app options passed as the first argument to initializeApp() for the " +
                ("app named \"" + this.name_ + "\". Options must be a non-null object."));
        }
        var hasCredential = ('credential' in this.options_);
        if (!hasCredential) {
            this.options_.credential = new credential_1.ApplicationDefaultCredential();
        }
        var credential = this.options_.credential;
        if (typeof credential !== 'object' || credential === null || typeof credential.getAccessToken !== 'function') {
            throw new error_1.FirebaseAppError(error_1.AppErrorCodes.INVALID_APP_OPTIONS, "Invalid Firebase app options passed as the first argument to initializeApp() for the " +
                ("app named \"" + this.name_ + "\". The \"credential\" property must be an object which implements ") +
                "the Credential interface.");
        }
        Object.keys(firebaseInternals_.serviceFactories).forEach(function (serviceName) {
            // Defer calling createService() until the service is accessed
            _this[serviceName] = _this.getService_.bind(_this, serviceName);
        });
        this.INTERNAL = new FirebaseAppInternals(this.options_.credential);
    }
    /**
     * Returns the Auth service instance associated with this app.
     *
     * @return {Auth} The Auth service instance of this app.
     */
    FirebaseApp.prototype.auth = function () {
        var _this = this;
        return this.ensureService_('auth', function () {
            var authService = require('./auth/auth').Auth;
            return new authService(_this);
        });
    };
    /**
     * Returns the Database service for the specified URL, and the current app.
     *
     * @return {Database} The Database service instance of this app.
     */
    FirebaseApp.prototype.database = function (url) {
        var _this = this;
        var service = this.ensureService_('database', function () {
            var dbService = require('./database/database').DatabaseService;
            return new dbService(_this);
        });
        return service.getDatabase(url);
    };
    /**
     * Returns the Messaging service instance associated with this app.
     *
     * @return {Messaging} The Messaging service instance of this app.
     */
    FirebaseApp.prototype.messaging = function () {
        var _this = this;
        return this.ensureService_('messaging', function () {
            var messagingService = require('./messaging/messaging').Messaging;
            return new messagingService(_this);
        });
    };
    /**
     * Returns the Storage service instance associated with this app.
     *
     * @return {Storage} The Storage service instance of this app.
     */
    FirebaseApp.prototype.storage = function () {
        var _this = this;
        return this.ensureService_('storage', function () {
            var storageService = require('./storage/storage').Storage;
            return new storageService(_this);
        });
    };
    FirebaseApp.prototype.firestore = function () {
        var _this = this;
        var service = this.ensureService_('firestore', function () {
            var firestoreService = require('./firestore/firestore').FirestoreService;
            return new firestoreService(_this);
        });
        return service.client;
    };
    /**
     * Returns the InstanceId service instance associated with this app.
     *
     * @return {InstanceId} The InstanceId service instance of this app.
     */
    FirebaseApp.prototype.instanceId = function () {
        var _this = this;
        return this.ensureService_('iid', function () {
            var iidService = require('./instance-id/instance-id').InstanceId;
            return new iidService(_this);
        });
    };
    /**
     * Returns the ProjectManagement service instance associated with this app.
     *
     * @return {ProjectManagement} The ProjectManagement service instance of this app.
     */
    FirebaseApp.prototype.projectManagement = function () {
        var _this = this;
        return this.ensureService_('project-management', function () {
            var projectManagementService = require('./project-management/project-management').ProjectManagement;
            return new projectManagementService(_this);
        });
    };
    /**
     * Returns the SecurityRules service instance associated with this app.
     *
     * @return {SecurityRules} The SecurityRules service instance of this app.
     */
    FirebaseApp.prototype.securityRules = function () {
        var _this = this;
        return this.ensureService_('security-rules', function () {
            var securityRulesService = require('./security-rules/security-rules').SecurityRules;
            return new securityRulesService(_this);
        });
    };
    Object.defineProperty(FirebaseApp.prototype, "name", {
        /**
         * Returns the name of the FirebaseApp instance.
         *
         * @return {string} The name of the FirebaseApp instance.
         */
        get: function () {
            this.checkDestroyed_();
            return this.name_;
        },
        enumerable: true,
        configurable: true
    });
    Object.defineProperty(FirebaseApp.prototype, "options", {
        /**
         * Returns the options for the FirebaseApp instance.
         *
         * @return {FirebaseAppOptions} The options for the FirebaseApp instance.
         */
        get: function () {
            this.checkDestroyed_();
            return deep_copy_1.deepCopy(this.options_);
        },
        enumerable: true,
        configurable: true
    });
    /**
     * Deletes the FirebaseApp instance.
     *
     * @return {Promise<void>} An empty Promise fulfilled once the FirebaseApp instance is deleted.
     */
    FirebaseApp.prototype.delete = function () {
        var _this = this;
        this.checkDestroyed_();
        this.firebaseInternals_.removeApp(this.name_);
        this.INTERNAL.delete();
        return Promise.all(Object.keys(this.services_).map(function (serviceName) {
            return _this.services_[serviceName].INTERNAL.delete();
        })).then(function () {
            _this.services_ = {};
            _this.isDeleted_ = true;
        });
    };
    FirebaseApp.prototype.ensureService_ = function (serviceName, initializer) {
        this.checkDestroyed_();
        var service;
        if (serviceName in this.services_) {
            service = this.services_[serviceName];
        }
        else {
            service = initializer();
            this.services_[serviceName] = service;
        }
        return service;
    };
    /**
     * Returns the service instance associated with this FirebaseApp instance (creating it on demand
     * if needed). This is used for looking up monkeypatched service instances.
     *
     * @param {string} serviceName The name of the service instance to return.
     * @return {FirebaseServiceInterface} The service instance with the provided name.
     */
    FirebaseApp.prototype.getService_ = function (serviceName) {
        this.checkDestroyed_();
        if (!(serviceName in this.services_)) {
            this.services_[serviceName] = this.firebaseInternals_.serviceFactories[serviceName](this, this.extendApp_.bind(this));
        }
        return this.services_[serviceName];
    };
    /**
     * Callback function used to extend an App instance at the time of service instance creation.
     */
    FirebaseApp.prototype.extendApp_ = function (props) {
        deep_copy_1.deepExtend(this, props);
    };
    /**
     * Throws an Error if the FirebaseApp instance has already been deleted.
     */
    FirebaseApp.prototype.checkDestroyed_ = function () {
        if (this.isDeleted_) {
            throw new error_1.FirebaseAppError(error_1.AppErrorCodes.APP_DELETED, "Firebase app named \"" + this.name_ + "\" has already been deleted.");
        }
    };
    return FirebaseApp;
}());
exports.FirebaseApp = FirebaseApp;
