//
//  STPPaymentHandler.swift
//  StripePayments
//
//  Created by Cameron Sabol on 5/10/19.
//  Copyright © 2019 Stripe, Inc. All rights reserved.
//

import Foundation
import PassKit
import SafariServices
@_spi(STP) import StripeCore

#if canImport(Stripe3DS2)
import Stripe3DS2
#endif

/// `STPPaymentHandlerActionStatus` represents the possible outcomes of requesting an action by `STPPaymentHandler`. An action could be confirming and/or handling the next action for a PaymentIntent.
@objc public enum STPPaymentHandlerActionStatus: Int {
    /// The action succeeded.
    case succeeded
    /// The action was cancelled by the cardholder/user.
    case canceled
    /// The action failed. See the error code for more details.
    case failed
}

/// Error codes generated by `STPPaymentHandler`
@objc public enum STPPaymentHandlerErrorCode: Int {
    /// Indicates that the action requires an authentication method not recognized or supported by the SDK.
    @objc(STPPaymentHandlerUnsupportedAuthenticationErrorCode)
    case unsupportedAuthenticationErrorCode

    /// Indicates that the action requires an authentication app, but either the app is not installed or the request to switch to the app was denied.
    @objc(STPPaymentHandlerRequiredAppNotAvailableErrorCode)
    case requiredAppNotAvailable

    /// Attach a payment method to the PaymentIntent or SetupIntent before using `STPPaymentHandler`.
    @objc(STPPaymentHandlerRequiresPaymentMethodErrorCode)
    case requiresPaymentMethodErrorCode

    /// The PaymentIntent or SetupIntent status cannot be resolved by `STPPaymentHandler`.
    @objc(STPPaymentHandlerIntentStatusErrorCode)
    case intentStatusErrorCode

    /// The action timed out.
    @objc(STPPaymentHandlerTimedOutErrorCode)
    case timedOutErrorCode

    /// There was an error in the Stripe3DS2 SDK.
    @objc(STPPaymentHandlerStripe3DS2ErrorCode)
    case stripe3DS2ErrorCode

    /// The transaction did not authenticate (e.g. user entered the wrong code).
    @objc(STPPaymentHandlerNotAuthenticatedErrorCode)
    case notAuthenticatedErrorCode

    /// `STPPaymentHandler` does not support concurrent actions.
    @objc(STPPaymentHandlerNoConcurrentActionsErrorCode)
    case noConcurrentActionsErrorCode

    /// Payment requires a valid `STPAuthenticationContext`.  Make sure your presentingViewController isn't already presenting.
    @objc(STPPaymentHandlerRequiresAuthenticationContextErrorCode)
    case requiresAuthenticationContextErrorCode

    /// There was an error confirming the Intent.
    /// Inspect the `paymentIntent.lastPaymentError` or `setupIntent.lastSetupError` property.
    @objc(STPPaymentHandlerPaymentErrorCode)
    case paymentErrorCode

    /// The provided PaymentIntent of SetupIntent client secret does not match the expected pattern for client secrets.
    /// Make sure that your server is returning the correct value and that is being passed to `STPPaymentHandler`.
    @objc(STPPaymentHandlerInvalidClientSecret)
    case invalidClientSecret
}

/// Completion block typedef for use in `STPPaymentHandler` methods for Payment Intents.
public typealias STPPaymentHandlerActionPaymentIntentCompletionBlock = (
    STPPaymentHandlerActionStatus, STPPaymentIntent?, NSError?
) -> Void
/// Completion block typedef for use in `STPPaymentHandler` methods for Setup Intents.
public typealias STPPaymentHandlerActionSetupIntentCompletionBlock = (
    STPPaymentHandlerActionStatus, STPSetupIntent?, NSError?
) -> Void

/// `STPPaymentHandler` is a utility class that confirms PaymentIntents/SetupIntents and handles any authentication required, such as 3DS1/3DS2 for Strong Customer Authentication.
/// It can present authentication UI on top of your app or redirect users out of your app (to e.g. their banking app).
/// - seealso: https://stripe.com/docs/payments/3d-secure
public class STPPaymentHandler: NSObject {

    /// The error domain for errors in `STPPaymentHandler`.
    @objc public static let errorDomain = "STPPaymentHandlerErrorDomain"

    private var currentAction: STPPaymentHandlerActionParams?
    /// YES from when a public method is first called until its associated completion handler is called.
    /// This property guards against simultaneous usage of this class; only one "next action" can be handled at a time.
    private static var inProgress = false
    private var safariViewController: SFSafariViewController?

    /// Set this to true if you want a specific test to run the _canPresent code
    /// it will automatically toggle back to false after running the code once
    internal var checkCanPresentInTest: Bool = false

    /// The globally shared instance of `STPPaymentHandler`.
    @objc public static let sharedHandler: STPPaymentHandler = STPPaymentHandler()

    /// The globally shared instance of `STPPaymentHandler`.
    @objc
    public class func shared() -> STPPaymentHandler {
        return STPPaymentHandler.sharedHandler
    }

    @_spi(STP) public init(
        apiClient: STPAPIClient = .shared,
        threeDSCustomizationSettings: STPThreeDSCustomizationSettings =
            STPThreeDSCustomizationSettings(),
        formSpecPaymentHandler: FormSpecPaymentHandler? = nil
    ) {
        self.apiClient = apiClient
        self.threeDSCustomizationSettings = threeDSCustomizationSettings
        self.formSpecPaymentHandler = formSpecPaymentHandler
        super.init()
    }

    /// By default `sharedHandler` initializes with STPAPIClient.shared.
    @objc public var apiClient: STPAPIClient

    /// Customizable settings to use when performing 3DS2 authentication.
    /// Note: Configure this before calling any methods.
    /// Defaults to `STPThreeDSCustomizationSettings()`.
    @objc public var threeDSCustomizationSettings: STPThreeDSCustomizationSettings

    internal var _simulateAppToAppRedirect: Bool = false

    /// When this flag is enabled, STPPaymentHandler will confirm certain PaymentMethods using
    /// Safari instead of SFSafariViewController. If you'd like to use this in your own
    /// testing or Continuous Integration platform, please see the IntegrationTester app
    /// for usage examples.
    ///
    /// Note: This flag is only intended for development, and only impacts payments made with testmode keys.
    /// Setting this to `true` with a livemode key will fail.
    @objc public var simulateAppToAppRedirect: Bool
    {
        get {
            _simulateAppToAppRedirect && STPAPIClient.shared.isTestmode
        }
        set {
            _simulateAppToAppRedirect = newValue
        }
    }

    private var formSpecPaymentHandler: FormSpecPaymentHandler?

    internal var _redirectShim: ((URL, URL?, Bool) -> Void)?

    /// Confirms the PaymentIntent with the provided parameters and handles any `nextAction` required
    /// to authenticate the PaymentIntent.
    /// Call this method if you are using automatic confirmation.  - seealso:https://stripe.com/docs/payments/accept-a-payment?platform=ios&ui=custom
    /// - Parameters:
    ///   - paymentParams: The params used to confirm the PaymentIntent. Note that this method overrides the value of `paymentParams.useStripeSDK` to `@YES`.
    ///   - authenticationContext: The authentication context used to authenticate the payment.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the PaymentIntent status is not necessarily STPPaymentIntentStatusSucceeded (e.g. some bank payment methods take days before the PaymentIntent succeeds).
    @objc(confirmPayment:withAuthenticationContext:completion:)
    public func confirmPayment(
        _ paymentParams: STPPaymentIntentParams,
        with authenticationContext: STPAuthenticationContext,
        completion: @escaping STPPaymentHandlerActionPaymentIntentCompletionBlock
    ) {
        if Self.inProgress {
            completion(.failed, nil, _error(for: .noConcurrentActionsErrorCode))
            return
        } else if !STPPaymentIntentParams.isClientSecretValid(paymentParams.clientSecret) {
            completion(.failed, nil, _error(for: .invalidClientSecret))
            return
        }
        Self.inProgress = true
        weak var weakSelf = self
        // wrappedCompletion ensures we perform some final logic before calling the completion block.
        let wrappedCompletion: STPPaymentHandlerActionPaymentIntentCompletionBlock = {
            status,
            paymentIntent,
            error in
            guard let strongSelf = weakSelf else {
                return
            }
            // Reset our internal state
            Self.inProgress = false

            // Use spec first to determine if we are in a terminal state
            if status == .succeeded,
                self.formSpecPaymentHandler?.isPIStatusSpecFinishedForPostConfirmPIStatus(
                    paymentIntent: paymentIntent,
                    paymentHandler: self
                ) ?? false
            {
                completion(.succeeded, paymentIntent, nil)
                return
                //             Else ensure the .succeeded case returns a PaymentIntent in the expected state.
            } else if let paymentIntent = paymentIntent, status == .succeeded {
                let successIntentState =
                    paymentIntent.status == .succeeded || paymentIntent.status == .requiresCapture
                    || (paymentIntent.status == .processing
                        && STPPaymentHandler._isProcessingIntentSuccess(
                            for: paymentIntent.paymentMethod?.type ?? .unknown
                        )
                        || (paymentIntent.status == .requiresAction
                            && strongSelf.isNextActionSuccessState(
                                nextAction: paymentIntent.nextAction
                            )))

                if error == nil && successIntentState {
                    completion(.succeeded, paymentIntent, nil)
                } else {
                    assert(false, "Calling completion with invalid state")
                    completion(
                        .failed,
                        paymentIntent,
                        error ?? strongSelf._error(for: .intentStatusErrorCode)
                    )
                }
                return
            }
            completion(status, paymentIntent, error)
        }

        let confirmCompletionBlock: STPPaymentIntentCompletionBlock = { paymentIntent, error in
            guard let strongSelf = weakSelf else {
                return
            }
            if let paymentIntent = paymentIntent,
                error == nil
            {
                strongSelf._handleNextAction(
                    forPayment: paymentIntent,
                    with: authenticationContext,
                    returnURL: paymentParams.returnURL
                ) { status, completedPaymentIntent, completedError in
                    wrappedCompletion(status, completedPaymentIntent, completedError)
                }
            } else {
                wrappedCompletion(.failed, paymentIntent, error as NSError?)
            }
        }

        var params = paymentParams
        // We always set useStripeSDK = @YES in STPPaymentHandler
        if !(params.useStripeSDK?.boolValue ?? false) {
            params = paymentParams.copy() as! STPPaymentIntentParams
            params.useStripeSDK = NSNumber(value: true)
        }
        apiClient.confirmPaymentIntent(
            with: params,
            expand: ["payment_method"],
            completion: confirmCompletionBlock
        )
    }

    /// :nodoc:
    @available(
        *,
        deprecated,
        message: "Use confirmPayment(_:with:completion:) instead",
        renamed: "confirmPayment(_:with:completion:)"
    )
    public func confirmPayment(
        withParams: STPPaymentIntentParams,
        authenticationContext: STPAuthenticationContext,
        completion: @escaping STPPaymentHandlerActionPaymentIntentCompletionBlock
    ) {
        self.confirmPayment(withParams, with: authenticationContext, completion: completion)
    }

    /// Handles any `nextAction` required to authenticate the PaymentIntent.
    /// Call this method if you are using server-side confirmation.  - seealso: https://stripe.com/docs/payments/accept-a-payment?platform=ios&ui=custom
    /// - Parameters:
    ///   - paymentIntentClientSecret: The client secret of the PaymentIntent to handle next actions for.
    ///   - authenticationContext: The authentication context used to authenticate the payment.
    ///   - returnURL: An optional URL to redirect your customer back to after they authenticate or cancel in a webview. This should match the returnURL you specified during PaymentIntent confirmation.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the PaymentIntent status is not necessarily STPPaymentIntentStatusSucceeded (e.g. some bank payment methods take days before the PaymentIntent succeeds).
    @objc(handleNextActionForPayment:withAuthenticationContext:returnURL:completion:)
            public func handleNextAction(
        forPayment paymentIntentClientSecret: String,
        with authenticationContext: STPAuthenticationContext,
        returnURL: String?,
        completion: @escaping STPPaymentHandlerActionPaymentIntentCompletionBlock
    ) {
        if !STPPaymentIntentParams.isClientSecretValid(paymentIntentClientSecret) {
            completion(.failed, nil, _error(for: .invalidClientSecret))
            return
        }
        apiClient.retrievePaymentIntent(
            withClientSecret: paymentIntentClientSecret,
            expand: ["payment_method"]
        ) { [weak self] paymentIntent, error in
            guard let self else {
                return
            }
            if let paymentIntent = paymentIntent, error == nil {
                self.handleNextAction(for: paymentIntent, with: authenticationContext, returnURL: returnURL, completion: completion)
            } else {
                completion(.failed, paymentIntent, error as NSError?)
            }
        }
    }

    /// (Internal) Handles any `nextAction` required to authenticate the PaymentIntent.
    /// Call this method if you are using server-side confirmation.  - seealso: https://stripe.com/docs/payments/accept-a-payment?platform=ios&ui=custom
    /// - Parameters:
    ///   - paymentIntent: The PaymentIntent to handle next actions for.
    ///   - authenticationContext: The authentication context used to authenticate the payment.
    ///   - returnURL: An optional URL to redirect your customer back to after they authenticate or cancel in a webview. This should match the returnURL you specified during PaymentIntent confirmation.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the PaymentIntent status is not necessarily STPPaymentIntentStatusSucceeded (e.g. some bank payment methods take days before the PaymentIntent succeeds).
    /// - Note: The PaymentIntent must have been fetched with an expanded paymentMethod object (see how `handleNextAction(forPayment:)` does this).
    @_spi(STP) public func handleNextAction(
        for paymentIntent: STPPaymentIntent,
        with authenticationContext: STPAuthenticationContext,
        returnURL: String?,
        completion: @escaping STPPaymentHandlerActionPaymentIntentCompletionBlock
    ) {
        if Self.inProgress {
            assert(false, "Should not handle multiple payments at once.")
            completion(.failed, nil, _error(for: .noConcurrentActionsErrorCode))
            return
        }
        if paymentIntent.paymentMethodId != nil {
            assert(paymentIntent.paymentMethod != nil, "A PaymentIntent w/ attached paymentMethod must be retrieved w/ an expanded PaymentMethod")
        }
        Self.inProgress = true

        weak var weakSelf = self
        // wrappedCompletion ensures we perform some final logic before calling the completion block.
        let wrappedCompletion: STPPaymentHandlerActionPaymentIntentCompletionBlock = {
            status,
            paymentIntent,
            error in
            guard let strongSelf = weakSelf else {
                return
            }
            // Reset our internal state
            Self.inProgress = false
            // Ensure the .succeeded case returns a PaymentIntent in the expected state.
            if let paymentIntent = paymentIntent,
               status == .succeeded
            {
                let successIntentState =
                paymentIntent.status == .succeeded || paymentIntent.status == .requiresCapture || paymentIntent.status == .requiresConfirmation
                || (paymentIntent.status == .processing && STPPaymentHandler._isProcessingIntentSuccess(for: paymentIntent.paymentMethod?.type ?? .unknown))
                || (paymentIntent.status == .requiresAction && strongSelf.isNextActionSuccessState(nextAction: paymentIntent.nextAction))

                if error == nil && successIntentState {
                    completion(.succeeded, paymentIntent, nil)
                } else {
                    assert(false, "Calling completion with invalid state")
                    completion(
                        .failed,
                        paymentIntent,
                        error ?? strongSelf._error(for: .intentStatusErrorCode)
                    )
                }
                return
            }
            completion(status, paymentIntent, error)
        }

        if paymentIntent.status == .requiresConfirmation {
            // The caller forgot to confirm the paymentIntent on the backend before calling this method
            wrappedCompletion(
                .failed,
                paymentIntent,
                _error(
                    for: .intentStatusErrorCode,
                    userInfo: [
                        STPError.errorMessageKey:
                            "Confirm the PaymentIntent on the backend before calling handleNextActionForPayment:withAuthenticationContext:completion.",
                    ]
                )
            )
        } else {
            _handleNextAction(
                forPayment: paymentIntent,
                with: authenticationContext,
                returnURL: returnURL
            ) { status, completedPaymentIntent, completedError in
                wrappedCompletion(status, completedPaymentIntent, completedError)
            }
        }
    }

    /// Confirms the SetupIntent with the provided parameters and handles any `nextAction` required
    /// to authenticate the SetupIntent.
    /// - seealso: https://stripe.com/docs/payments/save-during-payment?platform=ios
    /// - Parameters:
    ///   - setupIntentConfirmParams: The params used to confirm the SetupIntent. Note that this method overrides the value of `setupIntentConfirmParams.useStripeSDK` to `@YES`.
    ///   - authenticationContext: The authentication context used to authenticate the SetupIntent.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the SetupIntent status will always be STPSetupIntentStatusSucceeded.
            @objc(confirmSetupIntent:withAuthenticationContext:completion:)
    public func confirmSetupIntent(
        _ setupIntentConfirmParams: STPSetupIntentConfirmParams,
        with authenticationContext: STPAuthenticationContext,
        completion: @escaping STPPaymentHandlerActionSetupIntentCompletionBlock
    ) {
        if Self.inProgress {
            assert(false, "Should not handle multiple payments at once.")
            completion(.failed, nil, _error(for: .noConcurrentActionsErrorCode))
            return
        } else if !STPSetupIntentConfirmParams.isClientSecretValid(
            setupIntentConfirmParams.clientSecret
        ) {
            completion(.failed, nil, _error(for: .invalidClientSecret))
            return
        }

        Self.inProgress = true
        weak var weakSelf = self
        // wrappedCompletion ensures we perform some final logic before calling the completion block.
        let wrappedCompletion: STPPaymentHandlerActionSetupIntentCompletionBlock = {
            status,
            setupIntent,
            error in
            guard let strongSelf = weakSelf else {
                return
            }
            // Reset our internal state
            Self.inProgress = false

            if status == .succeeded {
                // Ensure the .succeeded case returns a PaymentIntent in the expected state.
                if let setupIntent = setupIntent,
                    error == nil,
                    setupIntent.status == .succeeded
                        || (setupIntent.status == .requiresAction
                            && self.isNextActionSuccessState(nextAction: setupIntent.nextAction))
                {
                    completion(.succeeded, setupIntent, nil)
                } else {
                    assert(false, "Calling completion with invalid state")
                    completion(
                        .failed,
                        setupIntent,
                        error ?? strongSelf._error(for: .intentStatusErrorCode)
                    )
                }

            } else {
                completion(status, setupIntent, error)
            }
        }

        let confirmCompletionBlock: STPSetupIntentCompletionBlock = { setupIntent, error in
            guard let strongSelf = weakSelf else {
                return
            }

            if let setupIntent = setupIntent,
                error == nil
            {
                let action = STPPaymentHandlerSetupIntentActionParams(
                    apiClient: self.apiClient,
                    authenticationContext: authenticationContext,
                    threeDSCustomizationSettings: self.threeDSCustomizationSettings,
                    setupIntent: setupIntent,
                    returnURL: setupIntentConfirmParams.returnURL
                ) { status, resultSetupIntent, resultError in
                    guard let strongSelf2 = weakSelf else {
                        return
                    }
                    strongSelf2.currentAction = nil

                    wrappedCompletion(status, resultSetupIntent, resultError)
                }
                strongSelf.currentAction = action
                let requiresAction = strongSelf._handleSetupIntentStatus(forAction: action)
                if requiresAction {
                    strongSelf._handleAuthenticationForCurrentAction()
                }
            } else {
                wrappedCompletion(.failed, setupIntent, error as NSError?)
            }
        }
        var params = setupIntentConfirmParams
        if !(params.useStripeSDK?.boolValue ?? false) {
            params = setupIntentConfirmParams.copy() as! STPSetupIntentConfirmParams
            params.useStripeSDK = NSNumber(value: true)
        }
        apiClient.confirmSetupIntent(with: params, expand: ["payment_method"], completion: confirmCompletionBlock)
    }

    /// Handles any `nextAction` required to authenticate the SetupIntent.
    /// Call this method if you are confirming the SetupIntent on your backend and get a status of requires_action.
    /// - Parameters:
    ///   - setupIntentClientSecret: The client secret of the SetupIntent to handle next actions for.
    ///   - authenticationContext: The authentication context used to authenticate the SetupIntent.
    ///   - returnURL: An optional URL to redirect your customer back to after they authenticate or cancel in a webview. This should match the returnURL you specified during SetupIntent confirmation.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the SetupIntent status will always be  STPSetupIntentStatusSucceeded.
    @objc(handleNextActionForSetupIntent:withAuthenticationContext:returnURL:completion:)
    public func handleNextAction(
        forSetupIntent setupIntentClientSecret: String,
        with authenticationContext: STPAuthenticationContext,
        returnURL: String?,
        completion: @escaping STPPaymentHandlerActionSetupIntentCompletionBlock
    ) {
        if !STPSetupIntentConfirmParams.isClientSecretValid(setupIntentClientSecret) {
            completion(.failed, nil, _error(for: .invalidClientSecret))
            return
        }

        apiClient.retrieveSetupIntent(withClientSecret: setupIntentClientSecret, expand: ["payment_method"]) { [weak self] setupIntent, error in
            guard let self else {
                return
            }
            if let setupIntent, error == nil {
                self.handleNextAction(for: setupIntent, with: authenticationContext, returnURL: returnURL, completion: completion)
            } else {
                completion(.failed, setupIntent, error as NSError?)
            }
        }
    }

    /// Handles any `nextAction` required to authenticate the SetupIntent.
    /// Call this method if you are confirming the SetupIntent on your backend and get a status of requires_action.
    /// - Parameters:
    ///   - setupIntent: The SetupIntent to handle next actions for.
    ///   - authenticationContext: The authentication context used to authenticate the SetupIntent.
    ///   - returnURL: An optional URL to redirect your customer back to after they authenticate or cancel in a webview. This should match the returnURL you specified during SetupIntent confirmation.
    ///   - completion: The completion block. If the status returned is `STPPaymentHandlerActionStatusSucceeded`, the SetupIntent status will always be  STPSetupIntentStatusSucceeded.
    /// - Note: The SetupIntent must have been fetched with an expanded paymentMethod object (see how `handleNextAction(forPayment:)` does this).
    @_spi(STP) public func handleNextAction(
        for setupIntent: STPSetupIntent,
        with authenticationContext: STPAuthenticationContext,
        returnURL: String?,
        completion: @escaping STPPaymentHandlerActionSetupIntentCompletionBlock
    ) {
        if Self.inProgress {
            assert(false, "Should not handle multiple payments at once.")
            completion(.failed, nil, _error(for: .noConcurrentActionsErrorCode))
            return
        }
        if setupIntent.paymentMethodID != nil {
            assert(setupIntent.paymentMethod != nil, "A SetupIntent w/ attached paymentMethod must be retrieved w/ an expanded PaymentMethod")
        }

        Self.inProgress = true
        weak var weakSelf = self
        // wrappedCompletion ensures we perform some final logic before calling the completion block.
        let wrappedCompletion: STPPaymentHandlerActionSetupIntentCompletionBlock = {
            status,
            setupIntent,
            error in
            guard let strongSelf = weakSelf else {
                return
            }
            // Reset our internal state
            Self.inProgress = false

            if status == .succeeded {
                // Ensure the .succeeded case returns a PaymentIntent in the expected state.
                if let setupIntent = setupIntent,
                   error == nil,
                   setupIntent.status == .succeeded
                {
                    completion(.succeeded, setupIntent, nil)
                } else {
                    assert(false, "Calling completion with invalid state")
                    completion(
                        .failed,
                        setupIntent,
                        error ?? strongSelf._error(for: .intentStatusErrorCode)
                    )
                }

            } else {
                completion(status, setupIntent, error)
            }
        }

        if setupIntent.status == .requiresConfirmation {
            // The caller forgot to confirm the setupIntent on the backend before calling this method
            wrappedCompletion(
                .failed,
                setupIntent,
                _error(
                    for: .intentStatusErrorCode,
                    userInfo: [
                        STPError.errorMessageKey:
                            "Confirm the SetupIntent on the backend before calling handleNextActionForSetupIntent:withAuthenticationContext:completion.",
                    ]
                )
            )
        } else {
            _handleNextAction(
                for: setupIntent,
                with: authenticationContext,
                returnURL: returnURL
            ) { status, completedSetupIntent, completedError in
                wrappedCompletion(status, completedSetupIntent, completedError)
            }
        }
    }

    // MARK: - Private Helpers

    /// Depending on the PaymentMethod Type, after handling next action and confirming,
    /// we should either expect a success state on the PaymentIntent, or for certain asynchronous
    /// PaymentMethods like SEPA Debit, processing is considered a completed PaymentIntent flow
    /// because the funds can take up to 14 days to transfer from the customer's bank.
    class func _isProcessingIntentSuccess(for type: STPPaymentMethodType) -> Bool {
        switch type {
        // Asynchronous payment methods whose intent.status is 'processing' after handling the next action
        case .SEPADebit,
            .bacsDebit,  // Bacs Debit takes 2-3 business days
            .AUBECSDebit,
            .sofort,
            .USBankAccount:
            return true

        // Synchronous
        case .alipay,
            .card,
            .UPI,
            .iDEAL,
            .FPX,
            .cardPresent,
            .giropay,
            .EPS,
            .payPal,
            .przelewy24,
            .bancontact,
            .netBanking,
            .OXXO,
            .grabPay,
            .afterpayClearpay,
            .blik,
            .weChatPay,
            .boleto,
            .link,
            .klarna,
            .affirm,
            .linkInstantDebit,
            .cashApp,
            .paynow,
            .zip,
            .revolutPay,
            .mobilePay,
            .amazonPay,
            .alma,
            .konbini,
            .promptPay,
            .swish:
            return false

        case .unknown:
            return false

        @unknown default:
            return false
        }
    }

    func _handleNextAction(
        forPayment paymentIntent: STPPaymentIntent,
        with authenticationContext: STPAuthenticationContext,
        returnURL returnURLString: String?,
        completion: @escaping STPPaymentHandlerActionPaymentIntentCompletionBlock
    ) {
        guard paymentIntent.status != .requiresPaymentMethod else {
            // The caller forgot to attach a paymentMethod.
            completion(
                .failed,
                paymentIntent,
                _error(for: .requiresPaymentMethodErrorCode)
            )
            return
        }

        weak var weakSelf = self
        let action = STPPaymentHandlerPaymentIntentActionParams(
            apiClient: apiClient,
            authenticationContext: authenticationContext,
            threeDSCustomizationSettings: threeDSCustomizationSettings,
            paymentIntent: paymentIntent,
            returnURL: returnURLString
        ) { status, resultPaymentIntent, error in
            guard let strongSelf = weakSelf else {
                return
            }
            strongSelf.currentAction = nil
            completion(status, resultPaymentIntent, error)
        }
        currentAction = action
        let specHandledNextAction =
            self.formSpecPaymentHandler?.handleNextActionSpec(
                for: paymentIntent,
                action: action,
                paymentHandler: self
            ) ?? false
        if !specHandledNextAction {
            let requiresAction = _handlePaymentIntentStatus(forAction: action)
            if requiresAction {
                _handleAuthenticationForCurrentAction()
            }
        }
    }

            func _handleNextAction(
        for setupIntent: STPSetupIntent,
        with authenticationContext: STPAuthenticationContext,
        returnURL returnURLString: String?,
        completion: @escaping STPPaymentHandlerActionSetupIntentCompletionBlock
    ) {
        guard setupIntent.status != .requiresPaymentMethod else {
            // The caller forgot to attach a paymentMethod.
            completion(
                .failed,
                setupIntent,
                _error(for: .requiresPaymentMethodErrorCode)
            )
            return
        }

        weak var weakSelf = self
        let action = STPPaymentHandlerSetupIntentActionParams(
            apiClient: apiClient,
            authenticationContext: authenticationContext,
            threeDSCustomizationSettings: threeDSCustomizationSettings,
            setupIntent: setupIntent,
            returnURL: returnURLString
        ) { status, resultSetupIntent, resultError in
            guard let strongSelf = weakSelf else {
                return
            }
            strongSelf.currentAction = nil
            completion(status, resultSetupIntent, resultError)
        }
        currentAction = action
        let requiresAction = _handleSetupIntentStatus(forAction: action)
        if requiresAction {
            _handleAuthenticationForCurrentAction()
        }
    }

    /// Calls the current action's completion handler for the SetupIntent status, or returns YES if the status is ...RequiresAction.
    func _handleSetupIntentStatus(
        forAction action: STPPaymentHandlerSetupIntentActionParams
    )
        -> Bool
    {
        guard let setupIntent = action.setupIntent else {
            assert(false, "Calling _handleSetupIntentStatus without a setupIntent")
            return false
        }

        switch setupIntent.status {
        case .unknown:
            action.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: _error(
                    for: .intentStatusErrorCode,
                    userInfo: [
                        "STPSetupIntent": setupIntent.description,
                    ]
                )
            )

        case .requiresPaymentMethod:
            // If the user forgot to attach a PaymentMethod, they get an error before this point.
            // If confirmation fails (eg not authenticated, card declined) the SetupIntent transitions to this state.
            if let lastSetupError = setupIntent.lastSetupError {
                if lastSetupError.code == STPSetupIntentLastSetupError.CodeAuthenticationFailure {
                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(for: .notAuthenticatedErrorCode)
                    )
                } else if lastSetupError.type == .card {
                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(
                            for: .paymentErrorCode,
                            apiErrorCode: lastSetupError.code,
                            userInfo: [
                                NSLocalizedDescriptionKey: lastSetupError.message ?? "",
                            ]
                        )
                    )
                } else {
                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(for: .paymentErrorCode, apiErrorCode: lastSetupError.code)
                    )
                }
            } else {
                action.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(for: .paymentErrorCode)
                )
            }

        case .requiresConfirmation:
            action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)

        case .requiresAction:
            return true

        case .processing:
            action.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: _error(for: .intentStatusErrorCode)
            )

        case .succeeded:
            action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)

        case .canceled:
            action.complete(with: STPPaymentHandlerActionStatus.canceled, error: nil)

        @unknown default:
            fatalError()
        }
        return false
    }

    /// Calls the current action's completion handler for the PaymentIntent status, or returns YES if the status is ...RequiresAction.
    func _handlePaymentIntentStatus(
        forAction action: STPPaymentHandlerPaymentIntentActionParams
    )
        -> Bool
    {
        guard let paymentIntent = action.paymentIntent else {
            assert(false, "Calling _handlePaymentIntentStatus without a paymentIntent")
            return false
        }

        switch paymentIntent.status {
        case .unknown:
            action.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: _error(
                    for: .intentStatusErrorCode,
                    userInfo: [
                        "STPPaymentIntent": paymentIntent.description,
                    ]
                )
            )

        case .requiresPaymentMethod:
            // If the user forgot to attach a PaymentMethod, they get an error before this point.
            // If confirmation fails (eg not authenticated, card declined) the PaymentIntent transitions to this state.
            if let lastPaymentError = paymentIntent.lastPaymentError {
                if lastPaymentError.code
                    == STPPaymentIntentLastPaymentError.ErrorCodeAuthenticationFailure
                {
                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(for: .notAuthenticatedErrorCode)
                    )
                } else if lastPaymentError.type == .card {

                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(
                            for: .paymentErrorCode,
                            apiErrorCode: lastPaymentError.code,
                            userInfo: [
                                NSLocalizedDescriptionKey: lastPaymentError.message ?? "",
                            ]
                        )
                    )
                } else {
                    action.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(for: .paymentErrorCode, apiErrorCode: lastPaymentError.code)
                    )
                }
            } else {
                action.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(for: .paymentErrorCode)
                )
            }

        case .requiresConfirmation:
            action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)

        case .requiresAction:
            return true

        case .processing:
            if let type = paymentIntent.paymentMethod?.type,
                STPPaymentHandler._isProcessingIntentSuccess(for: type)
            {
                action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)
            } else {
                action.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(for: .intentStatusErrorCode)
                )
            }

        case .succeeded:
            action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)

        case .requiresCapture:
            action.complete(with: STPPaymentHandlerActionStatus.succeeded, error: nil)

        case .canceled:
            action.complete(with: STPPaymentHandlerActionStatus.canceled, error: nil)

        case .requiresSource:
            fatalError()
        case .requiresSourceAction:
            fatalError()
        }
        return false
    }

            func _handleAuthenticationForCurrentAction() {
        guard let currentAction = currentAction,
            let authenticationAction = currentAction.nextAction()
        else {
            return
        }

        switch authenticationAction.type {
        case .unknown:
            currentAction.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: _error(
                    for: .unsupportedAuthenticationErrorCode,
                    userInfo: [
                        "STPIntentAction": authenticationAction.description,
                    ]
                )
            )

        case .redirectToURL:
            if let redirectToURL = authenticationAction.redirectToURL {
                _handleRedirect(to: redirectToURL.url, withReturn: redirectToURL.returnURL)
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .alipayHandleRedirect:
            if let alipayHandleRedirect = authenticationAction.alipayHandleRedirect {
                _handleRedirect(
                    to: alipayHandleRedirect.nativeURL,
                    fallbackURL: alipayHandleRedirect.url,
                    return: alipayHandleRedirect.returnURL
                )
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .weChatPayRedirectToApp:
            if let weChatPayRedirectToApp = authenticationAction.weChatPayRedirectToApp {
                _handleRedirect(
                    to: weChatPayRedirectToApp.nativeURL,
                    fallbackURL: nil,
                    return: nil
                )
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .OXXODisplayDetails:
            if let hostedVoucherURL = authenticationAction.oxxoDisplayDetails?.hostedVoucherURL {
                self._handleRedirect(to: hostedVoucherURL, withReturn: nil)
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .boletoDisplayDetails:
            if let hostedVoucherURL = authenticationAction.boletoDisplayDetails?.hostedVoucherURL {
                self._handleRedirect(to: hostedVoucherURL, withReturn: nil)
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .useStripeSDK:
            if let useStripeSDK = authenticationAction.useStripeSDK {
                switch useStripeSDK.type {
                case .unknown:
                    currentAction.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: _error(
                            for: .unsupportedAuthenticationErrorCode,
                            userInfo: [
                                "STPIntentAction": authenticationAction.description,
                            ]
                        )
                    )

                case .threeDS2Fingerprint:
                    guard let threeDSService = currentAction.threeDS2Service else {
                        currentAction.complete(
                            with: STPPaymentHandlerActionStatus.failed,
                            error: _error(
                                for: .stripe3DS2ErrorCode,
                                userInfo: [
                                    "description": "Failed to initialize STDSThreeDS2Service.",
                                ]
                            )
                        )
                        return
                    }
                    var transaction: STDSTransaction?
                    var authRequestParams: STDSAuthenticationRequestParameters?

                    STDSSwiftTryCatch.try(
                        {
                            transaction = threeDSService.createTransaction(
                                forDirectoryServer: useStripeSDK.directoryServerID ?? "",
                                serverKeyID: useStripeSDK.directoryServerKeyID,
                                certificateString: useStripeSDK.directoryServerCertificate ?? "",
                                rootCertificateStrings: useStripeSDK.rootCertificateStrings ?? [],
                                withProtocolVersion: "2.1.0"
                            )

                            authRequestParams = transaction?.createAuthenticationRequestParameters()

                        },
                        catch: { exception in

                            STPAnalyticsClient.sharedClient
                                .log3DS2AuthenticationRequestParamsFailed(
                                    with: currentAction.apiClient._stored_configuration,
                                    intentID: currentAction.intentStripeID ?? "",
                                    error: self._error(
                                        for: .stripe3DS2ErrorCode,
                                        userInfo: [
                                            "exception": exception.description,
                                        ]
                                    )
                                )

                            currentAction.complete(
                                with: STPPaymentHandlerActionStatus.failed,
                                error: self._error(
                                    for: .stripe3DS2ErrorCode,
                                    userInfo: [
                                        "exception": exception.description,
                                    ]
                                )
                            )
                        },
                        finallyBlock: {
                        }
                    )

                    STPAnalyticsClient.sharedClient.log3DS2AuthenticateAttempt(
                        with: currentAction.apiClient._stored_configuration,
                        intentID: currentAction.intentStripeID ?? ""
                    )

                    if let authParams = authRequestParams,
                        let transaction = transaction
                    {
                        currentAction.threeDS2Transaction = transaction
                        currentAction.apiClient.authenticate3DS2(
                            authParams,
                            sourceIdentifier: useStripeSDK.threeDSSourceID ?? "",
                            returnURL: currentAction.returnURLString,
                            maxTimeout: currentAction.threeDSCustomizationSettings
                                .authenticationTimeout,
                            publishableKeyOverride: useStripeSDK.publishableKeyOverride
                        ) { (authenticateResponse, error) in
                            if let authenticateResponse = authenticateResponse,
                                error == nil
                            {

                                if let aRes = authenticateResponse.authenticationResponse {

                                    if aRes.isChallengeRequired {
                                        let challengeParameters = STDSChallengeParameters(
                                            authenticationResponse: aRes
                                        )

                                        let doChallenge: STPVoidBlock = {
                                            var presentationError: NSError?

                                            if !self._canPresent(
                                                with: currentAction.authenticationContext,
                                                error: &presentationError
                                            ) {
                                                currentAction.complete(
                                                    with: STPPaymentHandlerActionStatus.failed,
                                                    error: presentationError
                                                )
                                            } else {
                                                STDSSwiftTryCatch.try(
                                                    {
                                                        let presentingViewController = currentAction
                                                            .authenticationContext
                                                            .authenticationPresentingViewController()

                                                        if let paymentSheet =
                                                            presentingViewController
                                                            as? PaymentSheetAuthenticationContext
                                                        {
                                                            transaction.doChallenge(
                                                                with: challengeParameters,
                                                                challengeStatusReceiver: self,
                                                                timeout: TimeInterval(
                                                                    currentAction
                                                                        .threeDSCustomizationSettings
                                                                        .authenticationTimeout
                                                                        * 60
                                                                )
                                                            ) {
                                                                (
                                                                    threeDSChallengeViewController,
                                                                    completion
                                                                ) in
                                                                paymentSheet.present(
                                                                    threeDSChallengeViewController,
                                                                    completion: completion
                                                                )
                                                            }
                                                        } else {
                                                            transaction.doChallenge(
                                                                with: presentingViewController,
                                                                challengeParameters:
                                                                    challengeParameters,
                                                                challengeStatusReceiver: self,
                                                                timeout: TimeInterval(
                                                                    currentAction
                                                                        .threeDSCustomizationSettings
                                                                        .authenticationTimeout
                                                                        * 60
                                                                )
                                                            )
                                                        }

                                                    },
                                                    catch: { exception in
                                                        self.currentAction?.complete(
                                                            with: STPPaymentHandlerActionStatus
                                                                .failed,
                                                            error: self._error(
                                                                for: .stripe3DS2ErrorCode,
                                                                userInfo: [
                                                                    "exception": exception,
                                                                ]
                                                            )
                                                        )
                                                    },
                                                    finallyBlock: {
                                                    }
                                                )
                                            }
                                        }

                                        if currentAction.authenticationContext.responds(
                                            to: #selector(
                                                STPAuthenticationContext.prepare(forPresentation:))
                                        ) {
                                            currentAction.authenticationContext.prepare?(
                                                forPresentation: doChallenge
                                            )
                                        } else {
                                            doChallenge()
                                        }

                                    } else {
                                        // Challenge not required, finish the flow.
                                        transaction.close()
                                        currentAction.threeDS2Transaction = nil
                                        STPAnalyticsClient.sharedClient.log3DS2FrictionlessFlow(
                                            with: currentAction.apiClient._stored_configuration,
                                            intentID: currentAction.intentStripeID ?? ""
                                        )

                                        self._retrieveAndCheckIntentForCurrentAction()
                                    }

                                } else if let fallbackURL = authenticateResponse.fallbackURL {
                                    self._handleRedirect(
                                        to: fallbackURL,
                                        withReturn: URL(string: currentAction.returnURLString ?? "")
                                    )
                                } else {
                                    currentAction.complete(
                                        with: STPPaymentHandlerActionStatus.failed,
                                        error: self._error(
                                            for: .unsupportedAuthenticationErrorCode,
                                            userInfo: [
                                                "STPIntentAction": authenticationAction.description,
                                            ]
                                        )
                                    )
                                }

                            } else {
                                currentAction.complete(
                                    with: STPPaymentHandlerActionStatus.failed,
                                    error: (error as NSError?)
                                )
                            }
                        }

                    } else {
                        currentAction.complete(
                            with: STPPaymentHandlerActionStatus.failed,
                            error: self._error(
                                for: .unsupportedAuthenticationErrorCode,
                                userInfo: [
                                    "STPIntentAction": authenticationAction.description,
                                ]
                            )
                        )
                    }

                case .threeDS2Redirect:
                    if let redirectURL = useStripeSDK.redirectURL {
                        let returnURL: URL?
                        if let returnURLString = currentAction.returnURLString {
                            returnURL = URL(string: returnURLString)
                        } else {
                            returnURL = nil
                        }
                        _handleRedirect(to: redirectURL, withReturn: returnURL)
                    } else {
                        // TOOD : Error
                    }

                @unknown default:
                    fatalError()
                }
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }

        case .BLIKAuthorize:
            // The customer must authorize the transaction in their banking app within 1 minute
            // The merchant integration should spin and poll their backend or Stripe to determine success
            // If we are using PaymentSheet we want to use the PollingViewController, otherwise we should use the logic for API bindings users
            guard
                let presentingVC = currentAction.authenticationContext
                    as? PaymentSheetAuthenticationContext
            else {
                guard let currentAction = self.currentAction
                                    as? STPPaymentHandlerPaymentIntentActionParams
                else {
                    fatalError()
                }
                currentAction.complete(with: .succeeded, error: nil)
                return
            }
            presentingVC.presentPollingVCForAction(action: currentAction, type: .blik, safariViewController: nil)

        case .verifyWithMicrodeposits:
            // The customer must authorize after the microdeposits appear in their bank account
            // which may take 1-2 business days
            currentAction.complete(with: .succeeded, error: nil)

        case .upiAwaitNotification:
            guard
                let presentingVC = currentAction.authenticationContext
                    as? PaymentSheetAuthenticationContext
            else {
                return
            }

            presentingVC.presentPollingVCForAction(action: currentAction, type: .UPI, safariViewController: nil)
        case .cashAppRedirectToApp:
            guard
                let returnURL = URL(string: currentAction.returnURLString ?? "")
            else {
                fatalError()
            }

            if let mobileAuthURL = authenticationAction.cashAppRedirectToApp?.mobileAuthURL {
                _handleRedirect(to: mobileAuthURL, fallbackURL: mobileAuthURL, return: returnURL)
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }
        case .payNowDisplayQrCode:
            guard
                let returnURL = URL(string: currentAction.returnURLString ?? ""),
                let presentingVC = currentAction.authenticationContext
                    as? PaymentSheetAuthenticationContext,
                let hostedInstructionsURL = authenticationAction.payNowDisplayQrCode?.hostedInstructionsURL

            else {
                fatalError()
            }

            _handleRedirect(to: hostedInstructionsURL, fallbackURL: hostedInstructionsURL, return: returnURL) { safariViewController in
                // Present the polling view controller behind the web view so we can start polling right away
                presentingVC.presentPollingVCForAction(action: currentAction, type: .paynow, safariViewController: safariViewController)
            }
        case .konbiniDisplayDetails:
            if let hostedVoucherURL = authenticationAction.konbiniDisplayDetails?.hostedVoucherURL {
                self._handleRedirect(to: hostedVoucherURL, withReturn: nil)
            } else {
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: _error(
                        for: .unsupportedAuthenticationErrorCode,
                        userInfo: [
                            "STPIntentAction": authenticationAction.description,
                        ]
                    )
                )
            }
        case .promptpayDisplayQrCode:
            guard
                let returnURL = URL(string: currentAction.returnURLString ?? ""),
                let presentingVC = currentAction.authenticationContext
                    as? PaymentSheetAuthenticationContext,
                let hostedInstructionsURL = authenticationAction.promptPayDisplayQrCode?.hostedInstructionsURL

            else {
                fatalError()
            }

            _handleRedirect(to: hostedInstructionsURL, fallbackURL: hostedInstructionsURL, return: returnURL) { safariViewController in
                // Present the polling view controller behind the web view so we can start polling right away
                presentingVC.presentPollingVCForAction(action: currentAction, type: .promptPay, safariViewController: safariViewController)
            }
        case .swishHandleRedirect:
            guard
                let returnURL = URL(string: currentAction.returnURLString ?? ""),
                let mobileAuthURL = authenticationAction.swishHandleRedirect?.mobileAuthURL
            else {
                fatalError()
            }

            _handleRedirect(to: mobileAuthURL, withReturn: returnURL)
        @unknown default:
            fatalError()
        }
    }

    public func followRedirects(to url: URL, urlSession: URLSession) -> URL {
        let urlRequest = URLRequest(url: url)
        let blockingDataTaskSemaphore = DispatchSemaphore(value: 0)

        var resultingUrl = url
        let task = urlSession.dataTask(with: urlRequest) { _, response, error in
            defer {
                blockingDataTaskSemaphore.signal()
            }

            guard error == nil,
                let httpResponse = response as? HTTPURLResponse,
                (200...299).contains(httpResponse.statusCode),
                let responseURL = response?.url
            else {
                return
            }
            resultingUrl = responseURL
        }
        task.resume()
        blockingDataTaskSemaphore.wait()
        return resultingUrl
    }

    func _retryAfterDelay(retryCount: Int, block: @escaping STPVoidBlock) {
        // Add some backoff time:
        let delayTime = TimeInterval(3)

        DispatchQueue.main.asyncAfter(deadline: .now() + delayTime) {
            block()
        }
    }

    func _retrieveAndCheckIntentForCurrentAction(retryCount: Int = maxChallengeRetries) {
        // Alipay requires us to hit an endpoint before retrieving the PI, to ensure the status is up to date.
        let pingMarlinIfNecessary: ((STPPaymentHandlerPaymentIntentActionParams, @escaping STPVoidBlock) -> Void) = {
            currentAction,
            completionBlock in
            if let paymentMethod = currentAction.paymentIntent?.paymentMethod,
                paymentMethod.type == .alipay,
                let alipayHandleRedirect = currentAction.nextAction()?.alipayHandleRedirect,
                let alipayReturnURL = alipayHandleRedirect.marlinReturnURL
            {

                // Make a request to the return URL
                let request: URLRequest = URLRequest(url: alipayReturnURL)
                let task: URLSessionDataTask = URLSession.shared.dataTask(
                    with: request,
                    completionHandler: { _, _, _ in
                        completionBlock()
                    }
                )
                task.resume()
            } else {
                completionBlock()
            }
        }

        if let currentAction = self.currentAction as? STPPaymentHandlerPaymentIntentActionParams,
            let paymentIntent = currentAction.paymentIntent
        {
            pingMarlinIfNecessary(
                currentAction,
                {
                    currentAction.apiClient.retrievePaymentIntent(
                        withClientSecret: paymentIntent.clientSecret,
                        expand: ["payment_method"]
                    ) { retrievedPaymentIntent, error in
                        currentAction.paymentIntent = retrievedPaymentIntent
                        if let error = error {
                            currentAction.complete(
                                with: STPPaymentHandlerActionStatus.failed,
                                error: error as NSError?
                            )
                        } else {
                            // If the transaction is still unexpectedly processing, refresh the PaymentIntent
                            // This could happen if, for example, a payment is approved in an SFSafariViewController, the user closes the sheet, and the approval races with this fetch.
                            if let type = retrievedPaymentIntent?.paymentMethod?.type,
                                !STPPaymentHandler._isProcessingIntentSuccess(for: type),
                                retrievedPaymentIntent?.status == .processing && retryCount > 0
                            {
                                self._retryAfterDelay(retryCount: retryCount) {
                                    self._retrieveAndCheckIntentForCurrentAction(
                                        retryCount: retryCount - 1
                                    )
                                }
                            } else {
                                if self.formSpecPaymentHandler?.handlePostConfirmPIStatusSpec(
                                    for: currentAction.paymentIntent,
                                    action: currentAction,
                                    paymentHandler: self
                                ) ?? false {
                                    return
                                }
                                let requiresAction: Bool = self._handlePaymentIntentStatus(
                                    forAction: currentAction
                                )
                                if requiresAction {
                                    // If the status is still RequiresAction, the user exited from the redirect before the
                                    // payment intent was updated. Consider it a cancel, unless it's a valid terminal next action
                                    if self.isNextActionSuccessState(
                                        nextAction: retrievedPaymentIntent?.nextAction
                                    ) {
                                        currentAction.complete(with: .succeeded, error: nil)
                                    } else {
                                        // If this is a web-based 3DS2 transaction that is still in requires_action, we may just need to refresh the PI a few more times.
                                        // Also retry a few times for app redirects, the redirect flow is fast and sometimes the intent doesn't update quick enough
                                        let shouldRetryForCard = retrievedPaymentIntent?.paymentMethod?.type == .card && retrievedPaymentIntent?.nextAction?.type == .useStripeSDK
                                        let shouldRetryForAppRedirect = [.cashApp, .swish].contains(retrievedPaymentIntent?.paymentMethod?.type)
                                        if retryCount > 0
                                            && (shouldRetryForCard || shouldRetryForAppRedirect)
                                        {
                                            self._retryAfterDelay(retryCount: retryCount) {
                                                self._retrieveAndCheckIntentForCurrentAction(
                                                    retryCount: retryCount - 1
                                                )
                                            }
                                        } else if retrievedPaymentIntent?.paymentMethod?.type != .paynow
                                                    && retrievedPaymentIntent?.paymentMethod?.type != .promptPay {
                                            // For PayNow, we don't want to mark as canceled when the web view dismisses
                                            // Instead we rely on the presented PollingViewController to complete the currentAction
                                            self._markChallengeCanceled(withCompletion: { _, _ in
                                                // We don't forward cancelation errors
                                                currentAction.complete(
                                                    with: STPPaymentHandlerActionStatus.canceled,
                                                    error: nil
                                                )
                                            })
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            )
        } else if let currentAction = self.currentAction
            as? STPPaymentHandlerSetupIntentActionParams,
            let setupIntent = currentAction.setupIntent
        {

            currentAction.apiClient.retrieveSetupIntent(
                withClientSecret: setupIntent.clientSecret,
                expand: ["payment_method"]
            ) { retrievedSetupIntent, error in
                currentAction.setupIntent = retrievedSetupIntent
                if let error = error {
                    currentAction.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: error as NSError?
                    )
                } else {
                    if let type = retrievedSetupIntent?.paymentMethod?.type,
                        !STPPaymentHandler._isProcessingIntentSuccess(for: type),
                        retrievedSetupIntent?.status == .processing && retryCount > 0
                    {
                        self._retryAfterDelay(retryCount: retryCount) {
                            self._retrieveAndCheckIntentForCurrentAction(retryCount: retryCount - 1)
                        }
                    } else {
                        let requiresAction: Bool = self._handleSetupIntentStatus(
                            forAction: currentAction
                        )

                        if requiresAction {
                            // If the status is still RequiresAction, the user exited from the redirect before the
                            // payment intent was updated. Consider it a cancel, unless it's a valid terminal next action
                            if self.isNextActionSuccessState(
                                nextAction: retrievedSetupIntent?.nextAction
                            ) {
                                currentAction.complete(with: .succeeded, error: nil)
                            } else {
                                // If this is a web-based 3DS2 transaction that is still in requires_action, we may just need to refresh the SI a few more times.
                                // Also retry a few times for Cash App, the redirect flow is fast and sometimes the intent doesn't update quick enough
                                let shouldRetryForCard = retrievedSetupIntent?.paymentMethod?.type == .card && retrievedSetupIntent?.nextAction?.type == .useStripeSDK
                                let shouldRetryForCashApp = retrievedSetupIntent?.paymentMethod?.type == .cashApp
                                if retryCount > 0
                                    && (shouldRetryForCard || shouldRetryForCashApp) {
                                    self._retryAfterDelay(retryCount: retryCount) {
                                        self._retrieveAndCheckIntentForCurrentAction(
                                            retryCount: retryCount - 1
                                        )
                                    }
                                } else {
                                    // If the status is still RequiresAction, the user exited from the redirect before the
                                    // setup intent was updated. Consider it a cancel
                                    self._markChallengeCanceled(withCompletion: { _, _ in
                                        // We don't forward cancelation errors
                                        currentAction.complete(
                                            with: STPPaymentHandlerActionStatus.canceled,
                                            error: nil
                                        )
                                    })
                                }
                            }
                        }
                    }
                }

            }
        } else {
            assert(false, "currentAction is an unknown type or nil intent.")
        }
    }

            @objc func _handleWillForegroundNotification() {
        NotificationCenter.default.removeObserver(
            self,
            name: UIApplication.willEnterForegroundNotification,
            object: nil
        )
        STPURLCallbackHandler.shared().unregisterListener(self)
        _retrieveAndCheckIntentForCurrentAction()
    }

            @_spi(STP) public func _handleRedirect(to url: URL, withReturn returnURL: URL?) {
        _handleRedirect(to: url, fallbackURL: url, return: returnURL)
    }

            @_spi(STP) public func _handleRedirectToExternalBrowser(to url: URL, withReturn returnURL: URL?) {
        if let redirectShim = _redirectShim {
            redirectShim(url, returnURL, false)
        }
        guard let currentAction = currentAction else {
            assert(false, "Calling _handleRedirect without a currentAction")
            return
        }
        if let returnURL = returnURL {
            STPURLCallbackHandler.shared().register(self, for: returnURL)
        }
        STPAnalyticsClient.sharedClient.logURLRedirectNextAction(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? ""
        )

        // Setting universalLinksOnly to false will allow iOS to open https:// urls in an external browser, hopefully Safari.
        // The links are expected to be either universal links or Stripe owned URLs.
        // In the case that it is a stripe owned URL, the URL is expected to redirect our financial partners, at which point Safari can
        // redirect to a native app if the app has been installed.  If Safari is not the default browser, then users not be
        // automatically navigated to the native app.
        let options: [UIApplication.OpenExternalURLOptionsKey: Any] = [
            UIApplication.OpenExternalURLOptionsKey.universalLinksOnly: false,
        ]
        UIApplication.shared.open(
            url,
            options: options,
            completionHandler: { _ in
                NotificationCenter.default.addObserver(
                    self,
                    selector: #selector(self._handleWillForegroundNotification),
                    name: UIApplication.willEnterForegroundNotification,
                    object: nil
                )
            }
        )
    }

    /// Handles redirection to URLs using a native URL or a fallback URL and updates the current action.
    /// Redirects to an app if possible, if that fails opens the url in a web view
    /// - Parameters:
    ///     - nativeURL: A URL to be opened natively.
    ///     - fallbackURL: A secondary URL to be attempted if the native URL is not available.
    ///     - returnURL: The URL to be registered with the `STPURLCallbackHandler`.
    ///     - completion: A completion block invoked after the URL redirection is handled. The SFSafariViewController used is provided as an argument, if it was used for the redirect.
    func _handleRedirect(to nativeURL: URL?, fallbackURL: URL?, return returnURL: URL?, completion: ((SFSafariViewController?) -> Void)? = nil) {
        if let redirectShim = _redirectShim, let url = nativeURL ?? fallbackURL {
            redirectShim(url, returnURL, true)
        }

        // During testing, the completion block is not called since the `UIApplication.open` completion block is never invoked.
        // As a workaround we invoke the completion in a defer block if the _redirectShim is not nil to simulate presenting a web view
        defer {
            if _redirectShim != nil {
                completion?(nil)
            }
        }

        var url = nativeURL
        guard let currentAction = currentAction else {
            assert(false, "Calling _handleRedirect without a currentAction")
            return
        }

        if let returnURL = returnURL {
            STPURLCallbackHandler.shared().register(self, for: returnURL)
        }

        STPAnalyticsClient.sharedClient.logURLRedirectNextAction(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? ""
        )

        // Open the link in SafariVC
        let presentSFViewControllerBlock: (() -> Void) = {
            let context = currentAction.authenticationContext

            let presentingViewController = context.authenticationPresentingViewController()

            let doChallenge: STPVoidBlock = {
                var presentationError: NSError?
                guard self._canPresent(with: context, error: &presentationError) else {
                    currentAction.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: presentationError
                    )
                    return
                }

                if let fallbackURL = fallbackURL,
                    ["http", "https"].contains(fallbackURL.scheme)
                {
                    let safariViewController = SFSafariViewController(url: fallbackURL)
                    safariViewController.modalPresentationStyle = .overFullScreen
#if !canImport(CompositorServices)
                    safariViewController.dismissButtonStyle = .close
                    safariViewController.delegate = self
#endif
                    if context.responds(
                        to: #selector(STPAuthenticationContext.configureSafariViewController(_:))
                    ) {
                        context.configureSafariViewController?(safariViewController)
                    }
                    self.safariViewController = safariViewController
                    presentingViewController.present(safariViewController, animated: true, completion: {
                      completion?(safariViewController)
                    })
                } else {
                    currentAction.complete(
                        with: STPPaymentHandlerActionStatus.failed,
                        error: self._error(
                            for: .requiredAppNotAvailable,
                            userInfo: [
                                "STPIntentAction": currentAction.description,
                            ]
                        )
                    )
                }
            }
            if context.responds(to: #selector(STPAuthenticationContext.prepare(forPresentation:))) {
                context.prepare?(forPresentation: doChallenge)
            } else {
                doChallenge()
            }
        }

        // Redirect to an app
        // We don't want universal links to open up Safari, but we do want to allow custom URL schemes
        var options: [UIApplication.OpenExternalURLOptionsKey: Any] = [:]
        #if !targetEnvironment(macCatalyst)
        if let scheme = url?.scheme, scheme == "http" || scheme == "https" {
            options[UIApplication.OpenExternalURLOptionsKey.universalLinksOnly] = true
        }
        #endif

        // If we're simulating app-to-app redirects, we always want to open the URL in Safari instead of an in-app web view.
        // We'll tell Safari to open all URLs, not just universal links.
        // If we don't have a nativeURL, we should open the fallbackURL in Safari instead.
        if simulateAppToAppRedirect {
            options[UIApplication.OpenExternalURLOptionsKey.universalLinksOnly] = false
            url = nativeURL ?? fallbackURL
        }

        // We don't check canOpenURL before opening the URL because that requires users to pre-register the custom URL schemes
        if let url = url {
            UIApplication.shared.open(
                url,
                options: options,
                completionHandler: { success in
                    if !success {
                        // no app installed, launch safari view controller
                        presentSFViewControllerBlock()
                    } else {
                        completion?(nil)
                        NotificationCenter.default.addObserver(
                            self,
                            selector: #selector(self._handleWillForegroundNotification),
                            name: UIApplication.willEnterForegroundNotification,
                            object: nil
                        )
                    }
                }
            )
        } else {
            presentSFViewControllerBlock()
        }
    }

    /// Checks if authenticationContext.authenticationPresentingViewController can be presented on.
    /// @note Call this method after `prepareAuthenticationContextForPresentation:`
    func _canPresent(
        with authenticationContext: STPAuthenticationContext,
        error: inout NSError?
    )
        -> Bool
    {
        // Always allow in tests:
        if NSClassFromString("XCTest") != nil {
            if checkCanPresentInTest {
                checkCanPresentInTest.toggle()
            } else {
                return true
            }
        }

        let presentingViewController =
            authenticationContext.authenticationPresentingViewController()
        var canPresent = true
        var errorMessage: String?

        // Is it in the window hierarchy?
        if presentingViewController.viewIfLoaded?.window == nil {
            canPresent = false
            errorMessage =
                "authenticationPresentingViewController is not in the window hierarchy. You should probably return the top-most view controller instead."
        }

        // Is it already presenting something?
        if presentingViewController.presentedViewController != nil {
            canPresent = false
            errorMessage =
                "authenticationPresentingViewController is already presenting. You should probably dismiss the presented view controller in `prepareAuthenticationContextForPresentation`."
        }

        if !canPresent {
            error = _error(
                for: .requiresAuthenticationContextErrorCode,
                userInfo: errorMessage != nil
                    ? [
                        STPError.errorMessageKey: errorMessage ?? "",
                    ] : nil
            )
        }
        return canPresent
    }

    /// Check if the intent.nextAction is expected state after a successful on-session transaction
    /// e.g. for voucher-based payment methods like OXXO that require out-of-band payment
    func isNextActionSuccessState(nextAction: STPIntentAction?) -> Bool {
        if let nextAction = nextAction {
            switch nextAction.type {
            case .unknown,
                .redirectToURL,
                .useStripeSDK,
                .alipayHandleRedirect,
                .weChatPayRedirectToApp,
                .cashAppRedirectToApp,
                .payNowDisplayQrCode,
                .promptpayDisplayQrCode,
                .swishHandleRedirect:
                return false
            case .OXXODisplayDetails,
                .boletoDisplayDetails,
                .konbiniDisplayDetails,
                .verifyWithMicrodeposits,
                .BLIKAuthorize,
                .upiAwaitNotification:
                return true
            }
        }
        return false
    }

    // This is only called after web-redirects because native 3DS2 cancels go directly
    // to the ACS
    func _markChallengeCanceled(withCompletion completion: @escaping STPBooleanSuccessBlock) {
        guard let currentAction = currentAction,
            let nextAction = currentAction.nextAction()
        else {
            assert(false, "Calling _markChallengeCanceled without currentAction or nextAction.")
            return
        }

        var threeDSSourceID: String?
        switch nextAction.type {
        case .redirectToURL:
            threeDSSourceID = nextAction.redirectToURL?.threeDSSourceID
        case .useStripeSDK:
            threeDSSourceID = nextAction.useStripeSDK?.threeDSSourceID
        case .OXXODisplayDetails, .alipayHandleRedirect, .unknown, .BLIKAuthorize,
            .weChatPayRedirectToApp, .boletoDisplayDetails, .verifyWithMicrodeposits,
            .upiAwaitNotification, .cashAppRedirectToApp, .konbiniDisplayDetails, .payNowDisplayQrCode,
            .promptpayDisplayQrCode, .swishHandleRedirect:
            break
        @unknown default:
            fatalError()
        }

        guard let cancelSourceID = threeDSSourceID else {
            // If there's no threeDSSourceID, there's nothing for us to cancel
            completion(true, nil)
            return
        }

        if let currentAction = self.currentAction as? STPPaymentHandlerPaymentIntentActionParams,
            let paymentIntent = currentAction.paymentIntent
        {
            guard
                paymentIntent.paymentMethod?.card != nil || paymentIntent.paymentMethod?.link != nil
            else {
                // Only cancel 3DS auth on payment method types that support 3DS.
                completion(true, nil)
                return
            }

            STPAnalyticsClient.sharedClient.log3DS2RedirectUserCanceled(
                with: currentAction.apiClient._stored_configuration,
                intentID: currentAction.intentStripeID ?? ""
            )

            let intentID = nextAction.useStripeSDK?.threeDS2IntentOverride ?? paymentIntent.stripeId

            currentAction.apiClient.cancel3DSAuthentication(
                forPaymentIntent: intentID,
                withSource: cancelSourceID,
                publishableKeyOverride: nextAction.useStripeSDK?.publishableKeyOverride
            ) { retrievedPaymentIntent, error in
                currentAction.paymentIntent = retrievedPaymentIntent
                completion(retrievedPaymentIntent != nil, error)
            }
        } else if let currentAction = self.currentAction
            as? STPPaymentHandlerSetupIntentActionParams,
            let setupIntent = currentAction.setupIntent
        {
            guard setupIntent.paymentMethod?.card != nil || setupIntent.paymentMethod?.link != nil
            else {
                // Only cancel 3DS auth on payment method types that support 3DS.
                completion(true, nil)
                return
            }

            STPAnalyticsClient.sharedClient.log3DS2RedirectUserCanceled(
                with: currentAction.apiClient._stored_configuration,
                intentID: currentAction.intentStripeID ?? ""
            )

            let intentID = nextAction.useStripeSDK?.threeDS2IntentOverride ?? setupIntent.stripeID

            currentAction.apiClient.cancel3DSAuthentication(
                forSetupIntent: intentID,
                withSource: cancelSourceID,
                publishableKeyOverride: nextAction.useStripeSDK?.publishableKeyOverride
            ) { retrievedSetupIntent, error in
                currentAction.setupIntent = retrievedSetupIntent
                completion(retrievedSetupIntent != nil, error)
            }
        } else {
            assert(false, "currentAction is an unknown type or nil intent.")
        }
    }

    static let maxChallengeRetries = 5
    func _markChallengeCompleted(
        withCompletion completion: @escaping STPBooleanSuccessBlock,
        retryCount: Int = maxChallengeRetries
    ) {
        guard let currentAction = currentAction,
            let useStripeSDK = currentAction.nextAction()?.useStripeSDK,
            let threeDSSourceID = useStripeSDK.threeDSSourceID
        else {
            completion(false, nil)
            return
        }

        currentAction.apiClient.complete3DS2Authentication(
            forSource: threeDSSourceID,
            publishableKeyOverride: useStripeSDK.publishableKeyOverride
        ) {
            success,
            error in
            if success {
                if let paymentIntentAction = currentAction
                    as? STPPaymentHandlerPaymentIntentActionParams,
                    let paymentIntent = paymentIntentAction.paymentIntent
                {

                    currentAction.apiClient.retrievePaymentIntent(
                        withClientSecret: paymentIntent.clientSecret,
                        expand: ["payment_method"]
                    ) { retrievedPaymentIntent, retrieveError in
                        paymentIntentAction.paymentIntent = retrievedPaymentIntent
                        completion(retrievedPaymentIntent != nil, retrieveError)
                    }
                } else if let setupIntentAction = currentAction
                    as? STPPaymentHandlerSetupIntentActionParams,
                    let setupIntent = setupIntentAction.setupIntent
                {
                    currentAction.apiClient.retrieveSetupIntent(
                        withClientSecret: setupIntent.clientSecret,
                        expand: ["payment_method"]
                    ) { retrievedSetupIntent, retrieveError in
                        setupIntentAction.setupIntent = retrievedSetupIntent
                        completion(retrievedSetupIntent != nil, retrieveError)
                    }
                } else {
                    assert(false, "currentAction is an unknown type or nil intent.")
                }
            } else {
                // This isn't guaranteed to succeed if the ACS isn't ready yet.
                // Try it a few more times if it fails with a 400. (RUN_MOBILESDK-126)
                if retryCount > 0
                    && (error as NSError?)?.code == STPErrorCode.invalidRequestError.rawValue
                {
                    self._retryAfterDelay(
                        retryCount: retryCount,
                        block: {
                            self._markChallengeCompleted(
                                withCompletion: completion,
                                retryCount: retryCount - 1
                            )
                        }
                    )
                } else {
                    completion(success, error)
                }
            }
        }
    }

    // MARK: - Errors
    @_spi(STP) public func _error(
        for errorCode: STPPaymentHandlerErrorCode,
        apiErrorCode: String? = nil,
        userInfo additionalUserInfo: [AnyHashable: Any]? = nil
    ) -> NSError {
        var userInfo: [AnyHashable: Any] = additionalUserInfo ?? [:]
        switch errorCode {
        // 3DS(2) flow expected user errors
        case .notAuthenticatedErrorCode:
            userInfo[NSLocalizedDescriptionKey] = STPLocalizedString(
                "We are unable to authenticate your payment method. Please choose a different payment method and try again.",
                "Error when 3DS2 authentication failed (e.g. customer entered the wrong code)"
            )

        case .timedOutErrorCode:
            userInfo[NSLocalizedDescriptionKey] = STPLocalizedString(
                "Timed out authenticating your payment method -- try again",
                "Error when 3DS2 authentication timed out."
            )

        // PaymentIntent has an unexpected/unknown status
        case .intentStatusErrorCode:
            // The PI's status is processing or unknown
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey] ?? "The PaymentIntent status cannot be handled."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        case .unsupportedAuthenticationErrorCode:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "The SDK doesn't recognize the PaymentIntent action type."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        case .requiredAppNotAvailable:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "This PaymentIntent action requires an app, but the app is not installed or the request to open the app was denied."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        // Programming errors
        case .requiresPaymentMethodErrorCode:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "The PaymentIntent requires a PaymentMethod or Source to be attached before using STPPaymentHandler."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        case .noConcurrentActionsErrorCode:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "The current action is not yet completed. STPPaymentHandler does not support concurrent calls to its API."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        case .requiresAuthenticationContextErrorCode:
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        // Exceptions thrown from the Stripe3DS2 SDK. Other errors are reported via STPChallengeStatusReceiver.
        case .stripe3DS2ErrorCode:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey] ?? "There was an error in the Stripe3DS2 SDK."
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

        // Confirmation errors (eg card was declined)
        case .paymentErrorCode:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "There was an error confirming the Intent. Inspect the `paymentIntent.lastPaymentError` or `setupIntent.lastSetupError` property."

            userInfo[NSLocalizedDescriptionKey] =
                apiErrorCode.flatMap({ NSError.Utils.localizedMessage(fromAPIErrorCode: $0) })
                ?? userInfo[NSLocalizedDescriptionKey]
                ?? NSError.stp_unexpectedErrorMessage()

        // Client secret format error
        case .invalidClientSecret:
            userInfo[STPError.errorMessageKey] =
                userInfo[STPError.errorMessageKey]
                ?? "The provided Intent client secret does not match the expected client secret format. Make sure your server is returning the correct value and that is passed to `STPPaymentHandler`."
            userInfo[NSLocalizedDescriptionKey] =
                userInfo[NSLocalizedDescriptionKey] ?? NSError.stp_unexpectedErrorMessage()
        }
        return NSError(
            domain: STPPaymentHandler.errorDomain,
            code: errorCode.rawValue,
            userInfo: userInfo as? [String: Any]
        )
    }
}

#if !canImport(CompositorServices)
extension STPPaymentHandler: SFSafariViewControllerDelegate {
    // MARK: - SFSafariViewControllerDelegate
    /// :nodoc:
    @objc
    public func safariViewControllerDidFinish(_ controller: SFSafariViewController) {
        let context = currentAction?.authenticationContext
        if context?.responds(
            to: #selector(STPAuthenticationContext.authenticationContextWillDismiss(_:))
        ) ?? false {
            context?.authenticationContextWillDismiss?(controller)
        }
        safariViewController = nil
        STPURLCallbackHandler.shared().unregisterListener(self)
        _retrieveAndCheckIntentForCurrentAction()
    }
}
#endif

/// :nodoc:
@_spi(STP) extension STPPaymentHandler: STPURLCallbackListener {
    /// :nodoc:
    @_spi(STP) public func handleURLCallback(_ url: URL) -> Bool {
        // Note: At least my iOS 15 device, willEnterForegroundNotification is triggered before this method when returning from another app, which means this method isn't called because it unregisters from STPURLCallbackHandler.
        let context = currentAction?.authenticationContext
        if context?.responds(
            to: #selector(STPAuthenticationContext.authenticationContextWillDismiss(_:))
        ) ?? false,
            let safariViewController = safariViewController
        {
            context?.authenticationContextWillDismiss?(safariViewController)
        }

        NotificationCenter.default.removeObserver(
            self,
            name: UIApplication.willEnterForegroundNotification,
            object: nil
        )
        STPURLCallbackHandler.shared().unregisterListener(self)
        safariViewController?.dismiss(animated: true) {
            self.safariViewController = nil
        }
        _retrieveAndCheckIntentForCurrentAction()
        return true
    }
}

extension STPPaymentHandler {
    // MARK: - STPChallengeStatusReceiver
    /// :nodoc:
    @objc(transaction:didCompleteChallengeWithCompletionEvent:)
    dynamic func transaction(
        _ transaction: STDSTransaction,
        didCompleteChallengeWith completionEvent: STDSCompletionEvent
    ) {
        guard let currentAction = currentAction else {
            assert(false, "Calling didCompleteChallengeWith without currentAction.")
            return
        }
        let transactionStatus = completionEvent.transactionStatus
        STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowCompleted(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? "",
            uiType: transaction.presentedChallengeUIType
        )
        if transactionStatus == "Y" {
            _markChallengeCompleted(withCompletion: { _, _ in
                if let currentAction = self.currentAction
                    as? STPPaymentHandlerPaymentIntentActionParams
                {
                    let requiresAction = self._handlePaymentIntentStatus(forAction: currentAction)
                    if requiresAction {
                        assert(
                            false,
                            "3DS2 challenge completed, but the PaymentIntent is still requiresAction"
                        )
                        currentAction.complete(
                            with: STPPaymentHandlerActionStatus.failed,
                            error: self._error(for: .intentStatusErrorCode)
                        )
                    }
                } else if let currentAction = self.currentAction
                    as? STPPaymentHandlerSetupIntentActionParams
                {
                    let requiresAction = self._handleSetupIntentStatus(forAction: currentAction)
                    if requiresAction {
                        assert(
                            false,
                            "3DS2 challenge completed, but the SetupIntent is still requiresAction"
                        )
                        currentAction.complete(
                            with: STPPaymentHandlerActionStatus.failed,
                            error: self._error(for: .intentStatusErrorCode)
                        )
                    }
                }
            })
        } else {
            // going to ignore the rest of the status types because they provide more detail than we require
            _markChallengeCompleted(withCompletion: { _, _ in
                currentAction.complete(
                    with: STPPaymentHandlerActionStatus.failed,
                    error: self._error(
                        for: .notAuthenticatedErrorCode,
                        userInfo: [
                            "transaction_status": transactionStatus,
                        ]
                    )
                )
            })
        }
    }

    /// :nodoc:
    @objc(transactionDidCancel:)
    dynamic func transactionDidCancel(_ transaction: STDSTransaction) {
        guard let currentAction = currentAction else {
            assert(false, "Calling transactionDidCancel without currentAction.")
            return
        }

        STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowUserCanceled(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? "",
            uiType: transaction.presentedChallengeUIType
        )
        _markChallengeCompleted(withCompletion: { _, _ in
            // we don't forward cancelation errors
            currentAction.complete(with: STPPaymentHandlerActionStatus.canceled, error: nil)
        })
    }

    /// :nodoc:
    @objc(transactionDidTimeOut:)
    dynamic func transactionDidTimeOut(_ transaction: STDSTransaction) {
        guard let currentAction = currentAction else {
            assert(false, "Calling transactionDidTimeOut without currentAction.")
            return
        }

        STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowTimedOut(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? "",
            uiType: transaction.presentedChallengeUIType
        )
        _markChallengeCompleted(withCompletion: { _, _ in
            currentAction.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: self._error(for: .timedOutErrorCode)
            )
        })

    }

    /// :nodoc:
    @objc(transaction:didErrorWithProtocolErrorEvent:)
    dynamic func transaction(
        _ transaction: STDSTransaction,
        didErrorWith protocolErrorEvent: STDSProtocolErrorEvent
    ) {

        guard let currentAction = currentAction else {
            assert(false, "Calling didErrorWith protocolErrorEvent without currentAction.")
            return
        }

        _markChallengeCompleted(withCompletion: { _, _ in
            // Add localizedError to the 3DS2 SDK error
            let threeDSError = protocolErrorEvent.errorMessage.nsErrorValue() as NSError
            var userInfo = threeDSError.userInfo
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

            let localizedError = NSError(
                domain: threeDSError.domain,
                code: threeDSError.code,
                userInfo: userInfo
            )
            STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowErrored(
                with: currentAction.apiClient._stored_configuration,
                intentID: currentAction.intentStripeID ?? "",
                error: localizedError
            )
            currentAction.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: localizedError
            )
        })
    }

    /// :nodoc:
    @objc(transaction:didErrorWithRuntimeErrorEvent:)
    dynamic func transaction(
        _ transaction: STDSTransaction,
        didErrorWith runtimeErrorEvent: STDSRuntimeErrorEvent
    ) {

        guard let currentAction = currentAction else {
            assert(false, "Calling didErrorWith runtimeErrorEvent without currentAction.")
            return
        }

        _markChallengeCompleted(withCompletion: { _, _ in
            // Add localizedError to the 3DS2 SDK error
            let threeDSError = runtimeErrorEvent.nsErrorValue() as NSError
            var userInfo = threeDSError.userInfo
            userInfo[NSLocalizedDescriptionKey] = NSError.stp_unexpectedErrorMessage()

            let localizedError = NSError(
                domain: threeDSError.domain,
                code: threeDSError.code,
                userInfo: userInfo
            )

            STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowErrored(
                with: currentAction.apiClient._stored_configuration,
                intentID: currentAction.intentStripeID ?? "",
                error: localizedError
            )
            currentAction.complete(
                with: STPPaymentHandlerActionStatus.failed,
                error: localizedError
            )
        })
    }

    /// :nodoc:
    @objc(transactionDidPresentChallengeScreen:)
    dynamic func transactionDidPresentChallengeScreen(_ transaction: STDSTransaction) {

        guard let currentAction = currentAction else {
            assert(false, "Calling didErrorWith runtimeErrorEvent without currentAction.")
            return
        }

        STPAnalyticsClient.sharedClient.log3DS2ChallengeFlowPresented(
            with: currentAction.apiClient._stored_configuration,
            intentID: currentAction.intentStripeID ?? "",
            uiType: transaction.presentedChallengeUIType
        )
    }

    /// :nodoc:
    @objc(dismissChallengeViewController:forTransaction:)
    dynamic func dismiss(
        _ challengeViewController: UIViewController,
        for transaction: STDSTransaction
    ) {
        guard let currentAction = currentAction else {
            assert(false, "Calling didErrorWith runtimeErrorEvent without currentAction.")
            return
        }
        if let paymentSheet = currentAction.authenticationContext
            .authenticationPresentingViewController() as? PaymentSheetAuthenticationContext
        {
            paymentSheet.dismiss(challengeViewController, completion: nil)
        } else {
            challengeViewController.dismiss(animated: true, completion: nil)
        }
    }

    @_spi(STP) public func cancel3DS2ChallengeFlow() {
        guard let transaction = currentAction?.threeDS2Transaction else {
            assertionFailure()
            return
        }
        transaction.cancelChallengeFlow()
    }
}

/// Internal authentication context for PaymentSheet magic
@_spi(STP) public protocol PaymentSheetAuthenticationContext: STPAuthenticationContext {
    func present(_ authenticationViewController: UIViewController, completion: @escaping () -> Void)
    func dismiss(_ authenticationViewController: UIViewController, completion: (() -> Void)?)
    func presentPollingVCForAction(action: STPPaymentHandlerActionParams, type: STPPaymentMethodType, safariViewController: SFSafariViewController?)
}

@_spi(STP) public protocol FormSpecPaymentHandler {
    func isPIStatusSpecFinishedForPostConfirmPIStatus(
        paymentIntent: STPPaymentIntent?,
        paymentHandler: STPPaymentHandler
    ) -> Bool
    func handleNextActionSpec(
        for paymentIntent: STPPaymentIntent,
        action: STPPaymentHandlerPaymentIntentActionParams,
        paymentHandler: STPPaymentHandler
    ) -> Bool
    func handlePostConfirmPIStatusSpec(
        for paymentIntent: STPPaymentIntent?,
        action: STPPaymentHandlerPaymentIntentActionParams,
        paymentHandler: STPPaymentHandler
    ) -> Bool
}
